import { defineStore } from "pinia";
import { ref } from "vue";
import axios from 'axios';
import router from '../Router';
import { useSuspiciousLoginStore } from "@/Stores/SuspiciousLoginStore.js";

export const useAuthStore = defineStore("auth", () => {
    const token = ref(localStorage.getItem('token'));
    const refreshToken = ref(localStorage.getItem('refresh_token'));
    const user = ref(JSON.parse(localStorage.getItem('user')));
    const isAuth = ref(false);
    let suspiciousLoginPollingInterval = null;
    let sessionValidityPollingInterval = null;
    let suspiciousLoginShown = false; // Flag to prevent multiple popups

    function setToken(tokenValue) {
        localStorage.setItem('token', tokenValue);
        token.value = tokenValue;
    }
    function setRefreshToken(tokenValue) {
        localStorage.setItem('refresh_token', tokenValue);
        refreshToken.value = tokenValue;
    }
    function getToken() {
        // Always get from localStorage for reliability
        return localStorage.getItem('token');
    }
    function getRefreshToken() {
        return refreshToken?.value;
    }
    function setUser(userValue) {
        if(userValue != null) {
            console.log('Auth store: setUser called with:', userValue);
            console.log('Auth store: userValue.id:', userValue.id);
            console.log('Auth store: userValue.data?.id:', userValue.data?.id);
            
            localStorage.setItem('user', JSON.stringify(userValue));
            user.value = userValue;
            // Start polling for suspicious logins
            startSuspiciousLoginPolling();
        }
    }
    function setIsAuth(auth) {
        isAuth.value = auth;
    }
    async function checkToken() {
        try {
            const tokenAuth = 'Bearer ' + token.value;
            const { data } = await axios.get('/api/auth/verify', {
                headers: {
                    Authorization: tokenAuth
                }
            });
            return data;
        } catch (error) {
            if(error.response.status === 401) {
                logout();
                router.push('/');
            }else{
                console.log(error.response);
            }
        }
    }
    function logout() {
        localStorage.removeItem('token');
        localStorage.removeItem('refresh_token');
        localStorage.removeItem('user');
        token.value = '';
        refreshToken.value = '';
        user.value = '';
        isAuth.value = false;
        // Stop polling when logged out
        stopSuspiciousLoginPolling();
        // Reset flag when user logs out
        suspiciousLoginShown = false;
    }

    // Polling-based suspicious login detection
    function startSuspiciousLoginPolling() {
        console.log('Auth store: startSuspiciousLoginPolling called');
        console.log('Auth store: user.value:', user.value);
        console.log('Auth store: user.value?.id:', user.value?.id);
        console.log('Auth store: user.value?.data?.id:', user.value?.data?.id);
        
        let userId = user.value?.id || user.value?.data?.id;
        
        // If no user ID found, try to get it from JWT token
        if (!userId) {
            console.log('Auth store: No user ID available for polling. User object:', user.value);
            console.log('Auth store: User object keys:', Object.keys(user.value || {}));
            
            // Try to get user ID from API first
            axios.get('/api/auth/me')
                .then(response => {
                    console.log('Auth store: Got user from API:', response.data);
                    const apiUserId = response.data.id;
                    if (apiUserId) {
                        console.log('Auth store: Using user ID from API:', apiUserId);
                        // Add delay to ensure session is created
                        setTimeout(() => startPollingWithUserId(apiUserId), 2000);
                    } else {
                        // Fallback to JWT token
                        const token = getToken();
                        if (token) {
                            try {
                                const payload = JSON.parse(atob(token.split('.')[1]));
                                const jwtUserId = payload.sub;
                                if (jwtUserId) {
                                    console.log('Auth store: Using user ID from JWT:', jwtUserId);
                                    // Add delay to ensure session is created
                                    setTimeout(() => startPollingWithUserId(jwtUserId), 2000);
                                    return;
                                }
                            } catch (error) {
                                console.error('Auth store: Error decoding JWT:', error);
                            }
                        }
                        console.log('Auth store: Could not get user ID from API or JWT, stopping polling');
                    }
                })
                .catch(error => {
                    console.error('Auth store: Error getting user from API:', error);
                    // Fallback to JWT token
                    const token = getToken();
                    if (token) {
                        try {
                            const payload = JSON.parse(atob(token.split('.')[1]));
                            console.log('Auth store: JWT payload:', payload);
                            const jwtUserId = payload.sub;
                            if (jwtUserId) {
                                console.log('Auth store: Using user ID from JWT:', jwtUserId);
                                // Add delay to ensure session is created
                                setTimeout(() => startPollingWithUserId(jwtUserId), 2000);
                                return;
                            }
                        } catch (error) {
                            console.error('Auth store: Error decoding JWT:', error);
                        }
                    }
                    console.log('Auth store: Could not get user ID from API or JWT, stopping polling');
                });
            return;
        }

        // Add delay to ensure session is created
        setTimeout(() => startPollingWithUserId(userId), 5000);
    }

    function startPollingWithUserId(userId) {
        console.log('Auth store: Starting suspicious login polling for user:', userId);
        console.log('Auth store: Current token:', getToken());
        
        // Check every 60 seconds for suspicious sessions (less aggressive)
        suspiciousLoginPollingInterval = setInterval(async () => {
            console.log('Auth store: Polling for suspicious sessions...');
            try {
                // Check for suspicious sessions
                const response = await axios.get('/api/auth/check-suspicious-sessions', {
                    headers: {
                        'Authorization': `Bearer ${getToken()}`
                    }
                });
                
                console.log('Auth store: Polling response:', response.data);
                
                if (response.data.suspicious_sessions && response.data.suspicious_sessions.length > 0) {
                    console.log('Auth store: Found suspicious sessions via polling:', response.data.suspicious_sessions);
                    
                    // Only show popup if not already shown
                    if (!suspiciousLoginShown) {
                        const suspiciousLoginStore = useSuspiciousLoginStore();
                        // Use the first suspicious session ID
                        suspiciousLoginStore.trigger(response.data.suspicious_sessions[0].id);
                        suspiciousLoginShown = true; // Mark as shown
                        console.log('Auth store: Triggered suspicious login store with session ID:', response.data.suspicious_sessions[0].id);
                    } else {
                        console.log('Auth store: Suspicious login popup already shown, skipping');
                    }
                } else {
                    console.log('Auth store: No suspicious sessions found');
                    // Reset flag if no suspicious sessions found
                    suspiciousLoginShown = false;
                }
            } catch (error) {
                console.error('Auth store: Error checking suspicious sessions:', error);
                console.error('Auth store: Error response:', error.response?.data);
                
                // If session is invalid (401), logout the current user
                if (error.response?.status === 401) {
                    console.log('Auth store: Session invalid, logging out current user');
                    logout();
                    router.push('/');
                    return;
                }
                
                // If too many requests, stop polling
                if (error.response?.status === 429) {
                    console.log('Auth store: Too many requests, stopping polling');
                    stopSuspiciousLoginPolling();
                }
            }
        }, 30000); // Check every 60 seconds (less aggressive)
        
        // Start session validity polling
        startSessionValidityPolling();
    }
    
    function startSessionValidityPolling() {
        console.log('Auth store: Starting session validity polling');
        
        // Check every 30 seconds if current session is still valid
        sessionValidityPollingInterval = setInterval(async () => {
            console.log('Auth store: Checking session validity...');
            try {
                const response = await axios.get('/api/auth/check-session-validity', {
                    headers: {
                        'Authorization': `Bearer ${getToken()}`
                    }
                });
                console.log('Auth store: Session is still valid');
            } catch (error) {
                console.error('Auth store: Session validity check failed:', error);
                
                // If session is invalid (401), logout the current user
                if (error.response?.status === 401) {
                    console.log('Auth store: Session invalidated, logging out current user');
                    logout();
                    router.push('/');
                    return;
                }
                
                // If too many requests, stop polling
                if (error.response?.status === 429) {
                    console.log('Auth store: Too many requests, stopping session validity polling');
                    stopSessionValidityPolling();
                }
            }
        }, 10000); // Check every 30 seconds
    }
    
    function stopSessionValidityPolling() {
        if (sessionValidityPollingInterval) {
            clearInterval(sessionValidityPollingInterval);
            sessionValidityPollingInterval = null;
            console.log('Auth store: Stopped session validity polling');
        }
    }

    function stopSuspiciousLoginPolling() {
        if (suspiciousLoginPollingInterval) {
            clearInterval(suspiciousLoginPollingInterval);
            suspiciousLoginPollingInterval = null;
            console.log('Auth store: Stopped suspicious login polling');
        }
        // Also stop session validity polling
        stopSessionValidityPolling();
    }

    return {
        token,
        refreshToken,
        user,
        isAuth,
        setToken,
        setRefreshToken,
        getToken,
        getRefreshToken,
        setUser,
        setIsAuth,
        checkToken,
        logout,
        startSuspiciousLoginPolling,
        stopSuspiciousLoginPolling,
        startPollingWithUserId,
        startSessionValidityPolling,
        stopSessionValidityPolling,
        suspiciousLoginShown
    };
});
